import os
import sys
import pygame
from config import LARGURA_JANELA, ALTURA_JANELA, QUADROS_POR_SEGUNDO, COMPRIMENTO_NIVEL
from util_imagem import carregar_imagem
from camera import Camera
from jogador import Jogador
from inimigo import Inimigo, Inimigo2
from cenario import construir_cenario
from estado import EstadoJogo
from ui import desenhar_grade, botao_reiniciar_topo, tela_inicio, hud_tempo, tela_final


def criar_jogo(dir_arquivos: str):
    """Cria todas as entidades do jogo"""
    solidos, decoracoes = construir_cenario(dir_arquivos, ALTURA_JANELA, COMPRIMENTO_NIVEL)
    jogador = Jogador(posicao_inicial=(100, ALTURA_JANELA - 200), dir_arquivos=dir_arquivos)
    inimigos = [
        Inimigo((900, ALTURA_JANELA - 160), dir_arquivos, (780, 980)),
        Inimigo2((1300, ALTURA_JANELA - 160), dir_arquivos, (1200, 1500)),
        Inimigo((1850, ALTURA_JANELA - 160), dir_arquivos, (1750, 2000)),
        Inimigo2((2400, ALTURA_JANELA - 160), dir_arquivos, (2250, 2550)),
        Inimigo((3100, ALTURA_JANELA - 160), dir_arquivos, (2950, 3300)),
        Inimigo2((3750, ALTURA_JANELA - 160), dir_arquivos, (3600, 3900)),
        Inimigo((4450, ALTURA_JANELA - 160), dir_arquivos, (4300, 4550)),
        Inimigo2((5200, ALTURA_JANELA - 160), dir_arquivos, (5050, 5350)),
        Inimigo2((5600, ALTURA_JANELA - 160), dir_arquivos, (5500, 5800)),
        Inimigo((5900, ALTURA_JANELA - 160), dir_arquivos, (5850, 6000)),
    ]
    camera = Camera(largura=COMPRIMENTO_NIVEL, altura=ALTURA_JANELA, largura_janela=LARGURA_JANELA)
    return solidos, decoracoes, jogador, inimigos, camera


def processar_colisoes(jogador: Jogador, inimigos: list, estado: EstadoJogo):
    """Processa colisões entre jogador e inimigos"""
    if estado.estado == "jogando":
        for inimigo in list(inimigos):
            if jogador.retangulo.colliderect(inimigo.retangulo):
                # Se pisar em cima, remove inimigo; senão, reinicia jogador
                if jogador.velocidade_y > 0 and jogador.retangulo.bottom <= inimigo.retangulo.top + 10:
                    inimigo.retangulo.topleft = (COMPRIMENTO_NIVEL + 1000, -1000)
                    try:
                        inimigos.remove(inimigo)
                    except ValueError:
                        pass
                    jogador.velocidade_y = -12
                else:
                    jogador.retangulo.topleft = (100, ALTURA_JANELA - 200)
                    jogador.velocidade_x = 0
                    jogador.velocidade_y = 0


def renderizar_cenario(tela: pygame.Surface, camera: Camera, solidos: list, decoracoes: list, img_nuvens, nuvens: list):
    """Renderiza o cenário (fundo, nuvens, decorações, chão)"""
    cor_ceu = (135, 206, 235)
    tela.fill(cor_ceu)
    
    # Nuvens com paralaxe
    if img_nuvens:
        for x, y in nuvens:
            tela.blit(img_nuvens, (x - int(camera.deslocamento_x * 0.5), y))
    
    # Grade de referência
    desenhar_grade(tela, camera.deslocamento_x, COMPRIMENTO_NIVEL, ALTURA_JANELA)
    
    # Decorações (tijolos, canos)
    for img, ret in decoracoes:
        tela.blit(img, camera.aplicar(ret))
    
    # Chão (barra verde)
    cor_chao = (60, 179, 113)
    for solido in solidos:
        if solido.height >= 64 and solido.width > 500:
            pygame.draw.rect(tela, cor_chao, camera.aplicar(solido))


def executar_jogo():
    pygame.init()
    tela = pygame.display.set_mode((LARGURA_JANELA, ALTURA_JANELA))
    pygame.display.set_caption("Mario - Simples")
    relogio = pygame.time.Clock()

    # Configuração de arquivos
    dir_base = os.path.dirname(os.path.abspath(__file__))
    dir_arquivos = os.path.join(dir_base, "imagens")
    caminho_tempos = os.path.join(dir_base, "tempos.txt")
    caminho_nuvens = os.path.join(dir_arquivos, "nuvens.png")

    # Estado do jogo
    estado = EstadoJogo(caminho_tempos)
    solidos, decoracoes, jogador, inimigos, camera = criar_jogo(dir_arquivos)
    img_nuvens = carregar_imagem(caminho_nuvens, altura_escala=120) if os.path.exists(caminho_nuvens) else None
    nuvens = [(x, y) for x in range(0, COMPRIMENTO_NIVEL, 480) for y in (40, 90, 140)] if img_nuvens else []

    executando = True
    while executando:
        # Processamento de eventos
        for evento in pygame.event.get():
            if evento.type == pygame.QUIT:
                executando = False
            
            # Reiniciar com tecla R
            if evento.type == pygame.KEYDOWN and evento.key == pygame.K_r:
                solidos, decoracoes, jogador, inimigos, camera = criar_jogo(dir_arquivos)
                estado.reiniciar()
            
            # Clique no botão reiniciar do topo
            if evento.type == pygame.MOUSEBUTTONDOWN and evento.button == 1:
                mx, my = evento.pos
                btn_top = pygame.Rect(LARGURA_JANELA - 180, 10, 170, 36)
                if btn_top.collidepoint(mx, my):
                    solidos, decoracoes, jogador, inimigos, camera = criar_jogo(dir_arquivos)
                    estado.reiniciar()
            
            # Processar entrada específica do estado
            estado.processar_entrada(evento, caminho_tempos)

        # Atualizações do jogo
        if estado.estado == "jogando":
            jogador.atualizar(solidos)
            for inimigo in inimigos:
                inimigo.patrulhar()
        
        estado.atualizar_cronometro()
        estado.verificar_fim_jogo(jogador.retangulo.x)
        camera.seguir(jogador.retangulo)
        processar_colisoes(jogador, inimigos, estado)

        # Renderização
        renderizar_cenario(tela, camera, solidos, decoracoes, img_nuvens, nuvens)
        
        # Entidades (inimigos e jogador)
        for inimigo in inimigos:
            tela.blit(inimigo.imagem, camera.aplicar(inimigo.retangulo))
        tela.blit(jogador.imagem, camera.aplicar(jogador.retangulo))

        # HUD e telas
        btn_top = botao_reiniciar_topo(tela)
        if estado.estado == "inicio":
            tela_inicio(tela, estado.registros)
        elif estado.estado == "jogando":
            hud_tempo(tela, estado.tempo_decorrido_ms)
        elif estado.estado == "final":
            if estado.tempo_final_ms is None:
                estado.tempo_final_ms = 0
            btn_ret = tela_final(tela, estado.tempo_final_ms, estado.nome_jogador)

        pygame.display.flip()
        relogio.tick(QUADROS_POR_SEGUNDO)

    pygame.quit()
    sys.exit(0)


if __name__ == "__main__":
    executar_jogo()